package com.huawei.sis.demo.cn_demo;

import com.huawei.sis.bean.AuthInfo;
import com.huawei.sis.bean.SisConfig;
import com.huawei.sis.bean.SisConstant;
import com.huawei.sis.bean.request.RegisterVoiceRequest;
import com.huawei.sis.bean.response.RegisterVoiceResponse;
import com.huawei.sis.bean.response.VoicesResponse;
import com.huawei.sis.client.VcsClient;
import com.huawei.sis.exception.SisException;
import com.huawei.sis.util.IOUtils;
import com.huawei.sis.util.JsonUtils;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * 声音复刻-声音注册和查询Demo
 * <p>
 * Copyright 2026 Huawei Technologies Co.,Ltd.
 */
public class VcsRegisterAndRequest {
    /*
     * 1. 用户先录制一段声音，进行注册
     * 2. 录音数据，使用base64编码，大小不超过6MB。支持wav、mp3、m4a格式，采样率不小于16kHz，时长在5-25秒，支持单、双通道。
     * 2. 用户查询已经注册的声音信息(包括预置的声音）
     */
    private static final Logger log = LoggerFactory.getLogger(VcsRegisterAndRequest.class);


    // 认证用的ak和sk硬编码到代码中或者明文存储都有很大的安全风险，建议在配置文件或者环境变量中密文存放，使用时解密，确保安全；
    private String ak = System.getenv("HUAWEICLOUD_SDK_AK");
    private String sk = System.getenv("HUAWEICLOUD_SDK_SK");
    private String region = "";    // 区域，如cn-east-3
    private String projectId = ""; // 项目id，在我的凭证查看。参考https://support.huaweicloud.com/api-sis/sis_03_0008.html
    private String serviceEndPoint = "";  // 非必选，参考终端节点
    private String path = ""; // 用户录音的声音存放位置，文件格式为wav、mp3、m4a

    public static void main(String[] args) {
        VcsRegisterAndRequest vcsRegisterAndRequest = new VcsRegisterAndRequest();
        vcsRegisterAndRequest.RegisterAndRequest();
    }

    /**
     * 定义config，所有参数可选，设置超时时间等。
     *
     * @return SisConfig
     */
    private SisConfig getConfig() {
        SisConfig config = new SisConfig();
        // 设置连接超时，默认10000ms
        config.setConnectionTimeout(SisConstant.DEFAULT_CONNECTION_TIMEOUT);
        // 设置读取超时，默认10000ms
        config.setReadTimeout(SisConstant.DEFAULT_READ_TIMEOUT);
        // 设置代理, 一定要确保代理可用才启动此设置。 代理初始化也可用不加密的代理, new ProxyHostInfo(host, port);
        // 设置代理，可参考 config.setProxy(new ProxyHostInfo(host, port, user, password));
        return config;
    }

    /**
     * 声音复刻-声音注册和查询Demo
     */
    private void RegisterAndRequest() {
        // 两个接口可单独运行
        registerVoice(); // 声音注册接口
        listVoice();  // 声音查询接口
    }

    private void registerVoice() {
        try {

            // 1. 初始化AuthInfo
            // 定义authInfo，根据ak，sk，region，projectId
            AuthInfo authInfo = new AuthInfo(ak, sk, region, projectId);

            // 设置config，主要与超时有关
            SisConfig config = getConfig();
            VcsClient vcsClient = new VcsClient(authInfo, config);

            // 2. 配置请求
            String data = IOUtils.getEncodeDataByPath(path);
            String voiceName = "";  // 待注册的用户声音名称, 多次使用同一个名称注册, 会报错名称重复
            String language = "chinese"; // 待注册的用户声音语种
            RegisterVoiceRequest request = new RegisterVoiceRequest(voiceName, data, language);

            // 3. 发送请求，获取响应。
            RegisterVoiceResponse response = vcsClient.registerVoice(request);
            log.info(JsonUtils.obj2Str(response, true));
        } catch (SisException e) {
            log.info("error_code:" + e.getErrorCode() + " error_msg:" + e.getErrorMsg());
        }
    }

    private VoicesResponse listVoice() {
        try {

            // 1. 初始化AuthInfo
            // 定义authInfo，根据ak，sk，region，projectId
            AuthInfo authInfo = new AuthInfo(ak, sk, region, projectId, serviceEndPoint);

            // 设置config，主要与超时有关
            SisConfig config = getConfig();
            VcsClient vcsClient = new VcsClient(authInfo, config);

            // 2. 发送请求，获取响应。具体结果可通过response.getXX获取, 下面取前10个音色
            VoicesResponse response = vcsClient.query(10, 0);
            log.info(JsonUtils.obj2Str(response, true));
            return response;
        } catch (SisException e) {
            log.info("error_code:" + e.getErrorCode() + " error_msg:" + e.getErrorMsg());
            return null;
        }
    }
}
