package com.huawei.sis.demo.cn_demo;

import com.huawei.sis.bean.AuthInfo;
import com.huawei.sis.bean.SisConfig;
import com.huawei.sis.bean.SisConstant;
import com.huawei.sis.bean.request.TtsCustomRequest;
import com.huawei.sis.bean.response.TtsCustomResponse;
import com.huawei.sis.client.TtsCustomizationClient;
import com.huawei.sis.exception.SisException;
import com.huawei.sis.util.JsonUtils;

/**
 * 语音合成的demo
 *
 * Copyright 2021 Huawei Technologies Co.,Ltd.
 */
public class TtsDemo {

  private static final int DEFAULT_PITCH = 0;
  private static final int DEFAULT_SPEED = 0;
  private static final int DEFAULT_VOLUME = 50;

  // 认证用的ak和sk硬编码到代码中或者明文存储都有很大的安全风险，建议在配置文件或者环境变量中密文存放，使用时解密，确保安全；
  // 本示例以ak和sk保存在环境变量中来实现身份验证为例，运行本示例前请先在本地环境中设置环境变量HUAWEICLOUD_SDK_AK和HUAWEICLOUD_SDK_SK。
  private String ak = System.getenv("HUAWEICLOUD_SDK_AK");
  private String sk = System.getenv("HUAWEICLOUD_SDK_SK");
  private String region = "";    // 区域，如cn-north-1、cn-north-4
  private String projectId = ""; // 项目id，在我的凭证查看。参考https://support.huaweicloud.com/api-sis/sis_03_0008.html

  private String text = "";      // 待合成的文本
  private String path = "";      // 设置本地音频保存路径.可选择不保存到本地。需具体到文件，如D:/test.wav


  /**
   * 用于语音合成参数设置，例如发声人、音高、语速、音量、采样率、连接超时。所有参数均可以不设置，采用默认。
   *
   * @param request 语音合成请求
   */
  private void setParameter(TtsCustomRequest request) {

    // 设置语音格式，可选MP3，pcm等，详见api文档。默认wav
    request.setAudioFormat("wav");
    // 音高，[-500, 500], 默认0
    request.setPitch(DEFAULT_PITCH);
    // 语速，[-500, 500]，默认0
    request.setSpeed(DEFAULT_SPEED);
    // 音量，[0, 100]，默认50
    request.setVolume(DEFAULT_VOLUME);
    // 当前支持8000和16000，默认8000
    request.setSampleRate("8000");
    // 设置property，特征字符串，{language}_{speaker}_{domain}, 详见api文档
    request.setProperty("chinese_xiaoyu_common");

    // 设置返回数据是否保存，默认不保存。若保存，则需要设置一下保存路径，如D:/1.wav
    request.setSaved(true);
    request.setSavePath(path);
  }

  /**
   * 定义config，所有参数可选，设置超时时间等。
   *
   * @return SisConfig
   */
  private SisConfig getConfig() {
    SisConfig config = new SisConfig();
    // 设置连接超时，默认10000ms
    config.setConnectionTimeout(SisConstant.DEFAULT_CONNECTION_TIMEOUT);
    // 设置读取超时，默认10000ms
    config.setReadTimeout(SisConstant.DEFAULT_READ_TIMEOUT);
    // 设置代理, 一定要确保代理可用才启动此设置。 代理初始化也可用不加密的代理，new ProxyHostInfo(host, port);
    // ProxyHostInfo proxy = new ProxyHostInfo(host, port, username, password);
    // config.setProxy(proxy);
    return config;
  }

  /**
   * 根据文本和api，获取生成的音频数据
   */
  private void ttsCustomDemo() {
    try {
      // 1. 初始化TtsCustomizationClient
      // 定义authInfo，根据ak，sk，region, projectId.
      AuthInfo authInfo = new AuthInfo(ak, sk, region, projectId);
      // 定义config，所有参数可选，设置超时时间。
      SisConfig config = getConfig();
      // 根据authInfo和config，构造TtsCustomizationClient
      TtsCustomizationClient tts = new TtsCustomizationClient(authInfo, config);

      // 2. 配置请求
      TtsCustomRequest request = new TtsCustomRequest(text);
      // 设置参数，所有参数均可选，如果要保存合成音频文件，需要在request设置
      setParameter(request);

      // 3. 发送请求，获取响应。具体结果可通过response.getXX获取。
      TtsCustomResponse response = tts.getTtsResponse(request);
      System.out.println(JsonUtils.obj2Str(response, true));

    } catch (SisException e) {
      e.printStackTrace();
      System.out.println("error_code:" + e.getErrorCode() + "\nerror_msg:" + e.getErrorMsg());
    }

  }

  public static void main(String[] args) {
    TtsDemo demo = new TtsDemo();
    demo.ttsCustomDemo();
  }

}
