package com.huawei.sis.demo.cn_demo;

import com.huawei.sis.bean.SisConfig;
import com.huawei.sis.bean.SisConstant;
import com.huawei.sis.bean.request.AsrCustomLongRequest;
import com.huawei.sis.bean.response.AsrCustomLongResponse;
import com.huawei.sis.bean.AuthInfo;
import com.huawei.sis.client.AsrCustomizationClient;
import com.huawei.sis.exception.SisException;
import com.huawei.sis.util.JsonUtils;

/**
 * 录音文件识别Demo
 *
 * Copyright 2021 Huawei Technologies Co.,Ltd.
 */
public class LasrDemo {
  private static final int SLEEP_TIME = 500;
  private static final int MAX_POLLING_NUMS = 1000;

  // 认证用的ak和sk硬编码到代码中或者明文存储都有很大的安全风险，建议在配置文件或者环境变量中密文存放，使用时解密，确保安全；
  // 本示例以ak和sk保存在环境变量中来实现身份验证为例，运行本示例前请先在本地环境中设置环境变量HUAWEICLOUD_SDK_AK和HUAWEICLOUD_SDK_SK。
  private String ak = System.getenv("HUAWEICLOUD_SDK_AK");
  private String sk = System.getenv("HUAWEICLOUD_SDK_SK");
  private String region = "";    // 区域，如cn-north-1、cn-north-4
  private String projectId = ""; // 项目id，在我的凭证查看。参考https://support.huaweicloud.com/api-sis/sis_03_0008.html

  /**
   *   todo 请正确填写音频格式和模型属性字符串
   *   1. 音频格式一定要相匹配.
   *      例如文件或者obs url是xx.wav, 格式是auto。具体参考api文档。
   *      例如音频是pcm格式，并且采样率为8k，则格式填写pcm8k16bit。
   *      如果返回audio_format is invalid 说明该文件格式不支持。具体支持哪些音频格式，需要参考api文档。
   *
   *   2. 音频采样率要与属性字符串的采样率要匹配。
   *      例如格式选择pcm16k16bit，属性字符串却选择chinese_8k_common, 则会返回'audio_format' is not match model
   *      例如wav本身是16k采样率，属性选择chinese_8k_common, 同样会返回'audio_format' is not match model
   */

  private String obsUrl = "";           // 音频文件obs链接，录音文件识别目前仅支持传入obs音频连接。目前仅支持同一用户，同一region
  private String obsAudioFormat = "";   // 文件格式，如auto等，详见api文档
  private String obsProperty = "";      // 属性字符串，language_sampleRate_domain, 如chinese_8k_common, 详见api文档



  /**
   * 设置录音文件识别参数，所有参数均有默认值，不配置也可使用
   *
   * @param request 录音文件识别请求
   */
  private void setLongParameter(AsrCustomLongRequest request) {
    // 设置否是添加标点，yes 或no， 默认是no
    request.setAddPunc("yes");
    // 设置是否将音频中数字转写为阿拉伯数字，yes or no，默认yes
    request.setDigitNorm("no");
    // 设置是否需要分析，默认为false。只有其为true时，声道、话者分离、情绪检查、语速才生效。目前仅支持8k模型
    request.setNeedAnalysis(true);
    // 设置声道，MONO/LEFT_AGENT/RIGHT_AGENT, 默认是单声道MONO
    request.setChannel("MONO");
    // 设置是否需要话者分离，若是，则识别结果包含role，默认true
    request.setDirization(true);
    // 设置是否需要情绪检测，默认ture
    request.setEmotion(true);
    // 设置是否需要语速检查，默认true
    request.setSpeed(true);
    // 设置回调地址，设置后音频转写结果将直接发送至回调地址。请务必保证地址可联通。目前不支持ip调用
    // request.setCallbackUrl("");
    // 设置热词id，详见api文档，若热词id不存在，则会报错
    // request.setVocabularyId("");
    // 设置是否需要word_info，yes or no, 默认no
    request.setNeedWordInfo("no");
  }

  /**
   * 定义config，所有参数可选，设置超时时间等。
   *
   * @return SisConfig
   */
  private SisConfig getConfig() {
    SisConfig config = new SisConfig();
    // 设置连接超时，默认10000ms
    config.setConnectionTimeout(SisConstant.DEFAULT_CONNECTION_TIMEOUT);
    // 设置读取超时，默认10000ms
    config.setReadTimeout(SisConstant.DEFAULT_READ_TIMEOUT);
    // 设置代理, 一定要确保代理可用才启动此设置。 代理初始化也可用不加密的代理，new ProxyHostInfo(host, port);
    // ProxyHostInfo proxy = new ProxyHostInfo(host, port, username, password);
    // config.setProxy(proxy);
    return config;
  }


  /**
   * 录音文件识别demo。
   */
  private void lasrDemo() {
    try {
      // 1. 初始化AsrCustomizationClient
      // 定义authInfo，根据ak，sk，region,projectId.
      AuthInfo authInfo = new AuthInfo(ak, sk, region, projectId);
      // 设置config，主要与超时有关
      SisConfig config = getConfig();
      // 根据authInfo和config，构造AsrCustomizationClient
      AsrCustomizationClient asr = new AsrCustomizationClient(authInfo, config);

      // 2. 生成请求
      AsrCustomLongRequest request = new AsrCustomLongRequest(obsUrl, obsAudioFormat, obsProperty);
      // 设置请求参数，所有参数均为可选
      setLongParameter(request);

      // 3. 提交任务，获取jobId
      String jobId = asr.submitJob(request);

      // 4 轮询jobId，获取最终结果。
      int count = 0;
      int successFlag = 0;
      AsrCustomLongResponse response = null;
      while (count < MAX_POLLING_NUMS) {
        System.out.println("正在进行第" + count + "次尝试");
        response = asr.getAsrLongResponse(jobId);
        String status = response.getStatus();
        if (status.equals("FINISHED")) {
          successFlag = 1;
          break;
        } else if (status.equals("ERROR")) {
          System.out.println("执行失败, 无法根据jobId获取结果");
          return;
        }
        try {
          Thread.sleep(SLEEP_TIME);
        } catch (InterruptedException e) {
          e.printStackTrace();
        }
        count++;
      }
      // 打印结果
      if (successFlag == 0) {
        System.out.println("已进行" + count + "次尝试，无法获取识别结果。 jobId为 " + jobId);
        return;
      }
      System.out.println(JsonUtils.obj2Str(response, true));
    } catch (SisException e) {
      e.printStackTrace();
      System.out.println("error_code:" + e.getErrorCode() + "\nerror_msg:" + e.getErrorMsg());
    }
  }

  public static void main(String[] args) {
    LasrDemo demo = new LasrDemo();
    demo.lasrDemo();
  }

}
