package com.huawei.sis.demo.cn_demo;

import com.huawei.sis.bean.AuthInfo;
import com.huawei.sis.bean.SisConfig;
import com.huawei.sis.bean.SisConstant;
import com.huawei.sis.bean.request.HotWordRequest;
import com.huawei.sis.bean.response.HotWordsResponse;
import com.huawei.sis.bean.response.HotWordResponse;
import com.huawei.sis.client.HotWordClient;
import com.huawei.sis.exception.SisException;
import com.huawei.sis.util.JsonUtils;

import java.util.ArrayList;
import java.util.List;

/**
 * 热词demo
 *
 * 热词可在一句话识别、录音文件识别、实时语音识别使用。例如将地名和人名作为热词，则语音可以准确识别出人名和地名。
 * Copyright 2021 Huawei Technologies Co.,Ltd.
 */
public class HotWordDemo {
  // 认证用的ak和sk硬编码到代码中或者明文存储都有很大的安全风险，建议在配置文件或者环境变量中密文存放，使用时解密，确保安全；
  // 本示例以ak和sk保存在环境变量中来实现身份验证为例，运行本示例前请先在本地环境中设置环境变量HUAWEICLOUD_SDK_AK和HUAWEICLOUD_SDK_SK。
  private String ak = System.getenv("HUAWEICLOUD_SDK_AK");
  private String sk = System.getenv("HUAWEICLOUD_SDK_SK");
  private String region = "";    // 区域，如cn-north-1、cn-north-4
  private String projectId = ""; // 项目id，在我的凭证查看。参考https://support.huaweicloud.com/api-sis/sis_03_0008.html

  private String name = "test";      // 创建热词表时，需要确保热词表名之前未创建过。如 test1
  private String vocabularyId = "";  // 热词表id，仅在更新、查询，删除中使用,创建时不需要。使用前一定要确保热词表id已存在。
  private List<String> hotWordList = new ArrayList<>(); // 用于存放热词表，每个热词表最多存放1024个热词。如["计算机", "网络"]

  /**
   * 定义config，所有参数可选，设置超时时间等。
   *
   * @return SisConfig
   */
  private SisConfig getConfig() {
    SisConfig config = new SisConfig();
    // 设置连接超时，默认10000ms
    config.setConnectionTimeout(SisConstant.DEFAULT_CONNECTION_TIMEOUT);
    // 设置读取超时，默认10000ms
    config.setReadTimeout(SisConstant.DEFAULT_READ_TIMEOUT);
    // 设置代理, 一定要确保代理可用才启动此设置。 代理初始化也可用不加密的代理，new ProxyHostInfo(host, port);
    // ProxyHostInfo proxy = new ProxyHostInfo(host, port, username, password);
    // config.setProxy(proxy);
    return config;
  }

  /**
   * 1. 热词使用包含创建、更新、查询、删除等，一个用户可以创建多个热词表，一个热词表可以包含多个热词。一个vocabularyId对应一个热词表。
   * 2. 目前支持一个用户最多创建10个热词表，一个热词表最多包含1024个热词。
   */
  private void hotWordDemo() {
    try {
      // 初始化client
      AuthInfo authInfo = new AuthInfo(ak, sk, region, projectId);
      SisConfig config = getConfig();
      HotWordClient hotWordClient = new HotWordClient(authInfo, config);

      // option 1 创建热词表,可生成热词表id
      // name表示热词表表名，创建热词表时不可和已有表名重复
      // hotWordList表示热词表信息，用于存放热词
      hotWordList.add("测试");
      HotWordRequest hotWordRequest = new HotWordRequest(name, hotWordList);
      // 可选，热词表描述信息
      hotWordRequest.setDescription("test");
      // 可选，热词语言，目前仅支持中文 chinese_mandarin。
      hotWordRequest.setLanguage("chinese_mandarin");
      vocabularyId = hotWordClient.create(hotWordRequest);
      System.out.println("成功创建热词表，热词表id为" + vocabularyId);

      // option 2 根据热词表id 更新热词表。新的热词表会替换旧的热词表。使用前需确保热词表id已存在。
      hotWordList.add("华为");
      HotWordRequest updateRequest = new HotWordRequest(name, hotWordList);
      String updateVocabularyId = hotWordClient.update(updateRequest, vocabularyId);
      System.out.println("成功更新热词表，热词表id为" + updateVocabularyId);

      // option 3 查看热词表列表
      HotWordsResponse hotWordListResponse = hotWordClient.query();
      // 打印响应，可通过getXX方法获取具体参数
      System.out.println(JsonUtils.obj2Str(hotWordListResponse, true));

      // option 4 根据热词表id 查询热词表，使用前需确保热词表id已存在。。
      HotWordResponse hotWordResponse = hotWordClient.query(vocabularyId);
      // 打印响应，可通过getXX方法获取具体参数
      System.out.println(JsonUtils.obj2Str(hotWordResponse, true));

      // option 5 根据vocabularyId 删除热词表，使用前需确保热词表id已存在。
      // 删除无返回结果，无报错异常即表明删除成功。
      hotWordClient.delete(vocabularyId);
    } catch (SisException e) {
      e.printStackTrace();
      System.out.println("error_code:" + e.getErrorCode() + "\terror_msg:" + e.getErrorMsg());
    }

  }

  public static void main(String[] args) {
    HotWordDemo hotWordDemo = new HotWordDemo();
    hotWordDemo.hotWordDemo();
  }
}
